/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	CollectableItem
**
** DESCRIPTION:	Manages collectable items which are collected by the player
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef COLLECTABLE_ITEM_H
#define COLLECTABLE_ITEM_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include "ldwImage.h"
#include "CollectableInterface.h"

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/
const int CollectableItem_c_iMaxCollectables = 2;
const int CollectableItem_c_iMaxSpawnAreas = 8;

const int CollectableItem_c_iTimeOut = 90;

const int CollectableItem_c_iChanceOfCollectable = 3; //was 4 prior to 0.942
const int CollectableItem_c_iChanceOfUncommonCollectable = 17;//14
const int CollectableItem_c_iChanceOfRareCollectable = 4;//2

const int CollectableItem_c_iChanceOfShroom = 30;
const int CollectableItem_c_iChanceOfRareShroom = 8;

const int CollectableItem_c_iCommonCollectableTechValue = 100;
const int CollectableItem_c_iUncommonCollectableTechValue = 250;
const int CollectableItem_c_iRareCollectableTechValue = 1500;

const int CollectableItem_c_iNumCollections = eCarrying_NumberOf - eCarrying_BeginCollectables;

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class CCollectableItem : public ICollectable
{
private:

	struct SCollectable
	{
		bool			m_bExists;
		ECarrying		m_eType;
		unsigned int	m_uiTimeout;
		ldwPoint		m_sPos;

		int				m_iCollector;
		unsigned int	m_uiCollectionTime;
	};

	struct SSpawnArea
	{
		ldwRect			m_sRect;
		ECarrying		m_eType;
	};

public:

	//class structures
	struct SSaveState
	{
		int				m_iCollectionCount[CollectableItem_c_iNumCollections];

		SSpawnArea		m_sSpawnArea[CollectableItem_c_iMaxSpawnAreas];
		int				m_iNumSpawnAreas;
	};

private:

	//class data
	SCollectable		m_sCollectable[CollectableItem_c_iMaxCollectables];
	SSpawnArea			m_sSpawnArea[CollectableItem_c_iMaxSpawnAreas];
	int					m_iNumSpawnAreas;

	int					m_iCollectionCount[CollectableItem_c_iNumCollections];
	int					m_iNearestCollectable;
	ldwImageGrid *		m_pCollectables;

	//class methods
	//setup
	public:		CCollectableItem(void);
	public:		void Reset();
	public:		void LoadAssets();
	public:		void ReleaseAssets();
	public:		const bool LoadState(const SSaveState &sSaveState);
	public:		const bool SaveState(SSaveState &sSaveState) const;

	// process
	public:		void AddSpawnArea(const ldwRect sRect, const ECarrying eType);
	public:		void Add(const ECarrying eCarrying = eCarrying_Nothing, const ldwPoint sPos = ldwPoint(0, 0));
	public:		void AddRare();
	public: 	void AddTutorialItem();

	public:		void Remove(const int iCollectable);
	public:		void Update();
	public:		void Draw();
	public:		void Draw(const int iCollectable);

	public:		const bool Carry(CVillager &Villager, const ECarrying eCarrying);
	public:		void Drop(CVillager &Villager, const ECarrying eCarrying);

	// access
	public:		const ECarrying GetNearbyCollectable(CVillager &Villager, const ldwPoint sPos);
	public:		const bool IsCommonCollectable(const ECarrying eCarrying) const;
	public:		const bool IsUncommonCollectable(const ECarrying eCarrying) const;
	public:		const bool IsRareCollectable(const ECarrying eCarrying) const;
	public:		const bool IsCollectionComplete(const ECarrying eCarrying) const;
	public:		const int CollectionCount(const ECarrying eCarrying, const bool bIncludeCommon = true, const bool bIncludeUncommon = true, const bool bIncludeRare = true) const;
	public:		const int Count(const ECarrying eCarrying) const;
	public:		const bool WasItemSpawned(const ECarrying eCarrying) const;
	public:		const bool IsItemBeingCarried(const ECarrying eCarrying) const;
	public:		const bool WasTabletCollected(const ECarrying eCarrying) const;
};

/******************************************************************************\
** GLOBAL DECLARATIONS
\******************************************************************************/
extern CCollectableItem		CollectableItem;

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/